# MODAVA Authorization Matrix

This document outlines the complete authorization matrix showing who can perform what actions on which resources, including plan/feature-gate checks and middleware names.

## Role Definitions

- **superadmin**: Full system access, can manage all users and settings
- **admin**: Administrative access, can manage vendors and system settings
- **staff**: Limited admin access, can view and manage specific areas
- **vendor**: Business user, can manage their own data and customers

## Resource-Action Matrix

### Authentication & User Management

| Resource | Action | superadmin | admin | staff | vendor | Middleware | Feature Gate | Notes |
|----------|--------|------------|-------|-------|--------|------------|--------------|-------|
| User Account | Create | ✅ | ✅ | ❌ | ❌ | `requireAdmin` | None | Admin creation only |
| User Account | Read Own | ✅ | ✅ | ✅ | ✅ | `authenticateUser` | None | All authenticated users |
| User Account | Update Own | ✅ | ✅ | ✅ | ✅ | `authenticateUser` | None | Profile updates |
| User Account | Delete | ✅ | ✅ | ❌ | ❌ | `requireAdmin` | None | Admin deletion only |
| User Account | Impersonate | ✅ | ✅ | ❌ | ❌ | `requireAdmin` | None | Admin debugging tool |

### Vendor Management

| Resource | Action | superadmin | admin | staff | vendor | Middleware | Feature Gate | Notes |
|----------|--------|------------|-------|-------|--------|------------|--------------|-------|
| Vendor Profile | Read Own | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | None | Vendor access to own profile |
| Vendor Profile | Read Other | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | Admin can view all vendors |
| Vendor Profile | Update Own | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | None | Profile updates |
| Vendor Profile | Update Other | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | Admin management |
| Vendor Profile | Onboarding | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | None | Complete onboarding |
| Vendor Profile | Delete | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | Admin deletion only |

### Customer Management

| Resource | Action | superadmin | admin | staff | vendor | Middleware | Feature Gate | Notes |
|----------|--------|------------|-------|-------|--------|------------|--------------|-------|
| Customer | List | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `customer_management` | Plan-dependent feature |
| Customer | Create | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `customer_management` | Plan-dependent feature |
| Customer | Read | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `customer_management` | Plan-dependent feature |
| Customer | Update | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `customer_management` | Plan-dependent feature |
| Customer | Delete | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `customer_management` | Plan-dependent feature |
| Customer Measurements | Read | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `customer_management` | Plan-dependent feature |
| Customer Measurements | Update | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `customer_management` | Plan-dependent feature |

### Invoice Management

| Resource | Action | superadmin | admin | staff | vendor | Middleware | Feature Gate | Notes |
|----------|--------|------------|-------|-------|--------|------------|--------------|-------|
| Invoice | Create | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `invoicing_enabled` | Plan-dependent feature |
| Invoice | Read Own | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `invoicing_enabled` | Plan-dependent feature |
| Invoice | Read Public | ✅ | ✅ | ✅ | ✅ | None | None | Public invoice viewing |
| Invoice | Update | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `invoicing_enabled` | Plan-dependent feature |
| Invoice | Delete | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `invoicing_enabled` | Plan-dependent feature |
| Invoice | Download PDF | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `invoicing_enabled` | Plan-dependent feature |
| Invoice | Mark Paid | ✅ | ✅ | ✅ | ✅ | None | None | Webhook endpoint |
| Invoice Transactions | List | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `invoicing_enabled` | Plan-dependent feature |

### Reminder Management

| Resource | Action | superadmin | admin | staff | vendor | Middleware | Feature Gate | Notes |
|----------|--------|------------|-------|-------|--------|------------|--------------|-------|
| Reminder | Create | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `reminders` | Plan-dependent feature |
| Reminder | List | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `reminders` | Plan-dependent feature |
| Reminder | Send | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `reminders` | Plan-dependent feature |
| Reminder | Update | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `reminders` | Plan-dependent feature |
| Reminder | Delete | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | `reminders` | Plan-dependent feature |

### Subscription Management

| Resource | Action | superadmin | admin | staff | vendor | Middleware | Feature Gate | Notes |
|----------|--------|------------|-------|-------|--------|------------|--------------|-------|
| Subscription Plans | Read | ✅ | ✅ | ✅ | ✅ | None | None | Public endpoint |
| Subscription Plans | Create | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | Admin creation only |
| Subscription Plans | Update | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | Admin management only |
| Subscription Plans | Delete | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | Admin deletion only |
| Vendor Subscription | Read Own | ✅ | ✅ | ✅ | ✅ | `authenticateUser` | None | All authenticated users |
| Vendor Subscription | Activate | ✅ | ✅ | ✅ | ✅ | `authenticateUser` | None | Plan activation |
| Vendor Subscription | Trial Offer | ✅ | ✅ | ✅ | ✅ | `authenticateUser` | None | Trial management |
| Vendor Subscription | Accept Trial | ✅ | ✅ | ✅ | ✅ | `authenticateUser` | None | Trial acceptance |
| Vendor Features | Read Available | ✅ | ✅ | ✅ | ✅ | `authenticateUser` | None | Feature access check |

### KYC Management

| Resource | Action | superadmin | admin | staff | vendor | Middleware | Feature Gate | Notes |
|----------|--------|------------|-------|-------|--------|------------|--------------|-------|
| KYC Submission | Submit | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | None | Vendor submission |
| KYC Submission | Read Own Status | ✅ | ✅ | ✅ | ✅ | `authenticateUser` + `requireVendor` | None | Status check |
| KYC Submission | List Pending | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | Admin review |
| KYC Submission | Review | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | Admin approval |

### Market Pricing

| Resource | Action | superadmin | admin | staff | vendor | Middleware | Feature Gate | Notes |
|----------|--------|------------|-------|-------|--------|------------|--------------|-------|
| Price Report | Submit | ✅ | ✅ | ✅ | ✅ | `authenticateUser` | None | Market contribution |
| Price Report | List | ✅ | ✅ | ✅ | ✅ | `authenticateUser` | None | Market data access |
| Price Report | Suggestions | ✅ | ✅ | ✅ | ✅ | `authenticateUser` | None | Price recommendations |
| Pricing Access | Update | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | Admin control |

### Admin Functions

| Resource | Action | superadmin | admin | staff | vendor | Middleware | Feature Gate | Notes |
|----------|--------|------------|-------|-------|--------|------------|--------------|-------|
| Admin Users | List | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | Admin management |
| Admin Users | Create | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | Admin creation |
| Admin Users | Update Permissions | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | Permission management |
| Admin Settings | Read | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | System configuration |
| Admin Settings | Update | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | System configuration |
| System Logs | Read | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | System monitoring |
| Feature Flags | Update | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | Feature control |
| Audit Trail | Read | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | Compliance tracking |
| Support Tickets | List | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | Customer support |
| Support Tickets | Update | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | Ticket management |
| Admin Notifications | Read | ✅ | ✅ | ❌ | ❌ | `authenticateUser` + `requireAdmin` | None | System alerts |

### System Access

| Resource | Action | superadmin | admin | staff | vendor | Middleware | Feature Gate | Notes |
|----------|--------|------------|-------|-------|--------|------------|--------------|-------|
| Health Check | Read | ✅ | ✅ | ✅ | ✅ | None | None | Public endpoint |
| Rate Limiting | Bypass | ✅ | ✅ | ❌ | ❌ | Rate limit config | None | Admin rate limit exemption |

## Feature Gate Dependencies

### customer_management
- **Required for:** All customer CRUD operations
- **Plans:** Basic, Premium, Enterprise
- **Middleware:** `checkFeature('customer_management')`

### invoicing_enabled
- **Required for:** All invoice operations (except public viewing)
- **Plans:** Premium, Enterprise
- **Middleware:** `checkFeature('invoicing_enabled')`

### reminders
- **Required for:** All reminder operations
- **Plans:** Premium, Enterprise
- **Middleware:** `checkFeature('reminders')`

## Middleware Chain Order

1. **Rate Limiting** - Applied globally to all endpoints
2. **Authentication** - `authenticateUser` extracts and validates JWT token
3. **Role Check** - `requireVendor`, `requireAdmin`, etc. verify user permissions
4. **Feature Gate** - `checkFeature()` verifies plan-based feature access
5. **Validation** - `validateRequest` validates request body/params
6. **Controller** - Business logic execution

## Security Notes

- **JWT Tokens**: All authenticated endpoints require valid Bearer token
- **Row-Level Security**: Database enforces vendor isolation via RLS policies
- **Feature Gates**: Plan restrictions enforced at middleware level
- **Admin Access**: Admin functions require explicit admin role verification
- **Public Endpoints**: Only health check and invoice viewing are truly public
- **Webhook Security**: Payment webhooks should implement signature verification (not currently implemented)

## UNKNOWN Areas

The following authorization patterns are referenced but not fully documented:

- **Staff Role Permissions**: Exact scope of staff access unclear
- **Feature Flag Structure**: Complete feature flag schema not documented
- **Plan-Feature Mapping**: How plans map to specific features not detailed
- **Webhook Security**: Payment webhook authentication not implemented
- **Audit Logging**: Comprehensive audit trail implementation unclear

**File paths to investigate:**
- `src/middlewares/featureGate.ts`
- `src/modules/subscriptions/service.ts`
- `src/config/supabase.ts` (RLS policies)
